# 2017-Sep-10. Created by Maxim T. , 2024-09-11. Updated by @NewUser  
$scriptPath = Split-Path -parent $MyInvocation.MyCommand.Definition; 
. ($scriptPath + "\psFunctions.ps1");     # Adding script with reusable functions
. ($scriptPath + "\psSetVariables.ps1");  # Adding script to assign values to common variables

$logFile        = $scriptPath + "\Log\" + $MyInvocation.MyCommand.Name.Replace(".ps1",".txt"); 
(Get-Date).ToString("HH:mm:ss") + " --- Starting script " + $MyInvocation.MyCommand.Name | Out-File $logFile -Encoding OEM; # starting logging to file.
$logSummary = (Get-Date).ToString("HH:mm:ss") + " Script: Yahoo Historical".PadRight(28);
$quoteIDFile = $dataRootFolder + "\QuotesIntraDay\YahooIntraday.txt"; if (Test-Path $quoteIDFile) { Remove-Item $quoteIDFile;} # Removing intraday file before each load

$listStart = $config.IndexOf("<Yahoo>"); $listEnd = $config.IndexOf("</Yahoo>"); 
if ($listStart -eq -1 -or $listEnd -eq -1 -or $listStart+1 -ge $listEnd) {"<Yahoo> Symbol list is empty. Exiting script." | Out-File $logFile -Encoding OEM -Append; exit(1);}
$symbolList = @($config | Select-Object -Index(($listStart+1)..($listEnd-1))); #list of symbols we will work on
"Symbol count: " + $symbolList.count + ". MinDate: $minDate" | Out-File $logFile -Encoding OEM -Append;

$urlBase = "https://query2.finance.yahoo.com/v8/finance/chart/@Symbol@?period1=@FromDay@&period2=@ToDay@&interval=1d&events=history"
$urlBase = "https://query2.finance.yahoo.com/v8/finance/chart/@Symbol@?period1=@FromDay@&period2=@ToDay@&interval=1d&events=history"
$toDay   = [string] [math]::Floor((get-date -UFormat %s)); #today in unix timestamp
$urlBase = $urlBase.Replace("@ToDay@", $toDay);

ForEach($sLine in $symbolList) { # For each symbol
    $ret = GetSymbolInfo $sLine $quotesFolder $minDate; $symbol = $ret[0]; $nextDate = $ret[1]; $symbolMaxDate = $ret[2]; $symbolMaxDateAdj = $ret[3]; $symbolQuoteFile = $ret[4]; $lastQuote = $ret[5];
    "Symbol: $symbol. Next date: $nextDate. Quote file: $symbolQuoteFile" | Out-File $logFile -Encoding OEM -Append;
    if ($nextDate -gt $symbolMaxDate) { "  We already have data up to maximum configuration date of $symbolMaxDate. Will not request new data for this symbol." | Out-File $logFile -Encoding OEM -Append; continue; } 

    # ========== Get quotes from website - Start ==============
    $fromDay = [string] [math]::Floor((get-date $nextDate -UFormat %s));

    $url = $urlBase.Replace("@Symbol@",$symbol).Replace("@FromDay@",$fromDay);
    "  Requesting url: " + $url | Out-File $logFile -Encoding OEM -Append; 
	
	$webpage = ""; 
	$reqRows=0; 
	
    try {
		$reqCount++; $wr = Invoke-WebRequest -Uri $url -WebSession $websession -UseBasicParsing; #2025-12-10 Added -UseBasicParsing
	} catch { 
		# if attempt to get webpage failed go to next symbol
		$reqFailed++; "  " + $symbol + " - Not Found (web err) `r`n" | Out-File $logFile -Encoding OEM -Append; 
		continue; 
	} 
    if ($wr.StatusCode -ne 200) {
		$reqFailed++; "  " + $symbol + " - Returned status code: " + $wr.StatusCode + " `r`n" | Out-File $logFile -Encoding OEM -Append; 
		continue;  
	}

	# Fetch the JSON data
	$jsonData = Invoke-WebRequest -Uri $url -UseBasicParsing | Select-Object -ExpandProperty Content | ConvertFrom-Json #2025-12-10 Added -UseBasicParsing

	# Extract the relevant data
	$chartData = $jsonData.chart.result[0]
	$timestamp = $chartData.timestamp
	$closePrices = $chartData.indicators.quote[0].close
	$symbol = $chartData.meta.symbol

	$reqRows = 0
	
	# Combine the data into a custom object
	$data = for ($i = 0; $i -lt $timestamp.Count; $i++) {
		if ($closePrices[$i] -eq $null) {
			continue
		}
		[PSCustomObject]@{
			Date   = ([datetime]::FromFileTimeUtc($timestamp[$i] * 10000000 + 116444736000000000)).ToString("yyyy-MM-dd")
			Close  = ("{0:F6}" -f $closePrices[$i]).Replace(",",".") # If number has , then replace them with . (north american format)
			Symbol = $symbol
		}
		#$reqRows++
	}

    # filter results to include just dates we want
    $data = $data | Where-Object {$_.Date -ge $nextDate -and $_.Date -gt $lastQuote -and $_.Date -le $symbolMaxDateAdj -and $_.Date -lt $todayYMD -and $_.Symbol.Trim() -ne ""};
    $reqRows+= $data.Count;

	# Convert the data to CSV format without headers and quotes
	$csvData = $data | ConvertTo-Csv -NoTypeInformation | Select-Object -Skip 1
	$csvData = $csvData -replace '"', ''

	# Append the CSV data to the file
	$csvData | Out-File -FilePath $symbolQuoteFile -Append -Encoding OEM

	if ($recRows -eq 0) {
		"Done: $symbol. No new quotes found. `r`n" | Out-File $logFile -Encoding OEM -Append; # No new records where loaded
    } else {
		"Done: $symbol (from: $nextDate). Record count: $reqRows. Last quote: $lastQuote `r`n" | Out-File $logFile -Encoding OEM -Append; 
		$reqRowsT+=$reqRows
		$reqSucceed++
	}
}
$duration = (NEW-TIMESPAN -Start $startTime -End (Get-Date)).TotalSeconds.ToString("#,##0") + " sec.";
(Get-Date).ToString("HH:mm:ss") + " --- Finished. Quotes Requested/Succeed/Failed/Rows: $reqCount/$reqSucceed/$reqFailed/$reqRowsT. Duration: $duration`r`n" | Out-File $logFile -Encoding OEM -append;
$logSummary + ". Quotes Requested/Succeed/Failed/Rows: $reqCount/$reqSucceed/$reqFailed/$reqRowsT. Duration: $duration";
