# 2024-Apr-16. Created by Maxim T., 2024-09-11. Updated by @NewUser  

$scriptPath = Split-Path -parent $MyInvocation.MyCommand.Definition

# Adding script with reusable functions
. ($scriptPath + "\psFunctions.ps1")

# Adding script to assign values to common variables
. ($scriptPath + "\psSetVariables.ps1")

$logFile = $scriptPath + "\Log\" + $MyInvocation.MyCommand.Name.Replace(".ps1", ".txt")
(Get-Date).ToString("HH:mm:ss") + " --- Starting script " + $MyInvocation.MyCommand.Name | Out-File $logFile -Encoding OEM

# Starting logging to file
$logSummary = (Get-Date).ToString("HH:mm:ss") + " Script: Yahoo Intraday".PadRight(28)
$quoteIDFile = $dataRootFolder + "\QuotesIntraDay\YahooIntraday.txt"

# Removing intraday file before each load
if (Test-Path $quoteIDFile) {
    Remove-Item $quoteIDFile -Force
}

$symbolList = @()
$listStart = $config.IndexOf("<Yahoo>")
$listEnd = $config.IndexOf("</Yahoo>")

if ($listStart -eq -1 -or $listEnd -eq -1 -or $listStart + 1 -ge $listEnd) {
    "<Yahoo> Symbol list is empty." | Out-File $logFile -Encoding OEM -Append
    exit(1)
}

# List of symbols we will work on
$symbolList += @($config | Select-Object -Index(($listStart + 1)..($listEnd - 1)))

# By default if value is empty or y, then get intra-day quotes. Also MaxDate for symbol should not be specified
$symbolList = @($symbolList | ? {
    $a = $_.split(",")
    ($a[3] -eq $null -or $a[3] -eq "" -or $a[3] -eq "Y") -and ($a[2] -eq "" -or $a[2] -eq $null)
})

"Symbol count: " + $symbolList.count + ". MinDate: $minDate" | Out-File $logFile -Encoding OEM -Append

$urlBase = "https://query2.finance.yahoo.com/v8/finance/chart/@Symbol@?period1=@FromDay@&period2=@ToDay@&interval=1d&events=history&includeAdjustedClose=true"
$toDay = [string] [math]::Floor((get-date -hour 23 -Minute 59 -Second 59 -UFormat %s)) # Today in unix timestamp (end of the day)
$fromDay = [string] [math]::Floor((get-Date -hour 0 -Minute 0 -Second 0 -UFormat %s))  # Today in unix timestamp (start of the day)

# From and To dates are the same - today's date
$urlBase = $urlBase.Replace("@ToDay@", $toDay).Replace("@FromDay@", $fromDay)

$logSymbolFolder = $scriptPath + "\Log\SymbolWebpage"
if (!(Test-Path $logSymbolFolder)) {
    New-Item $logSymbolFolder -type directory
}

# Setting up folder where text output per symbol will be temporarily stored
ForEach ($sLine in $symbolList) {
    # For each symbol
    $ret = GetSymbolInfo $sLine $quotesFolder $minDate
    $symbol = $ret[0]
    $nextDate = $ret[1]
    $symbolMaxDate = $ret[2]
    $symbolMaxDateAdj = $ret[3]
    $symbolQuoteFile = $ret[4]
    $lastQuote = $ret[5]

    "Symbol: " + $symbol.PadLeft(12) + ". Next date: $nextDate. Quote file: $symbolQuoteFile" | Out-File $logFile -Encoding OEM -Append

    if ($nextDate -gt $symbolMaxDate) {
        "  We already have data up to maximum configuration date of $symbolMaxDate. Will not request new data for this symbol." | Out-File $logFile -Encoding OEM -Append
        continue
    }

    # ========== Get latest quote from website - Start ==============
    $reqRows = 0

    $url = $urlBase.Replace("@Symbol@", $symbol)
    "  Requesting url: " + $url | Out-File $logFile -Encoding OEM -Append

    try {
        $reqCount++
        $wr = Invoke-WebRequest -Uri $url -WebSession $websession -UseBasicParsing # 2025-12-10 Added -UseBasicParsing
    } catch {
        $reqFailed++
        "  " + $symbol + " - Not Found (web err) `r`n" | Out-File $logFile -Encoding OEM -Append
        continue
    }

    if ($wr.StatusCode -ne 200) {
        $reqFailed++
        "  " + $symbol + " - Returned status code: " + $wr.StatusCode + " `r`n" | Out-File $logFile -Encoding OEM -Append
        continue
    }
    $jsonObj = ConvertFrom-Json $wr.Content

    # Get the required data from the nested objects
    $date        = $jsonObj.chart.result[0].meta.regularMarketTime
    $symbol      = $jsonObj.chart.result[0].meta.symbol
    $regularPrice = $jsonObj.chart.result[0].meta.regularMarketPrice

    # Create an object with the required properties for output to CSV
    $outputObject = [PSCustomObject]@{
    	#Date        = [datetime]::UnixEpoch.AddSeconds($date).ToString("yyyy-MM-dd"); # Doesn't work on all systems
    	Date         = ([datetime]::FromFileTimeUtc($date * 10000000 + 116444736000000000)).ToString("yyyy-MM-dd")
    	RegularPrice = $regularPrice
    	Symbol       = $symbol
    }

    # Use "." for decimal and no 1000s grouping
    $culture = [System.Globalization.CultureInfo]::CreateSpecificCulture("en-US")
    $culture.NumberFormat.NumberDecimalSeparator = "."
    $culture.NumberFormat.NumberGroupSeparator = ""
    [System.Threading.Thread]::CurrentThread.CurrentCulture = $culture

    # Convert the data to CSV format without headers and quotes
    $csvData = $outputObject | ConvertTo-Csv -NoTypeInformation | Select-Object -Skip 1
    $csvData = $csvData -replace '"', ''

    # Append the CSV data to the file
    if ($outputObject.Date -ge $nextDate) {
        $csvData | Out-File -FilePath $quoteIDFile -Append -Encoding OEM
        $reqRows++
        $lastQuote = $outputObject.Date
    }

    $reqSucceed++

    if ($lastQuote -lt $nextDate) {
        "Done: $symbol. No new quotes found. `r`n" | Out-File $logFile -Encoding OEM -Append
    } else {
        "Done: $symbol (from: $nextDate). Record count: $reqRows. Last quote: $lastQuote `r`n" | Out-File $logFile -Encoding OEM -Append
        $reqRowsT += $reqRows
    }
} ## ForEach ($sLine in $symbolList) 

if (Test-Path -Path $quoteIDFile) {
    $fc = Get-Item -Path $quoteIDFile; if ($fc.Length -eq 0) {Remove-Item -Path $quoteIDFile;} # Removing file when size is 0;
}


$duration = (NEW-TIMESPAN -Start $startTime -End (Get-Date)).TotalSeconds.ToString("#,##0") + " sec."
(Get-Date).ToString("HH:mm:ss") + " --- Finished. Quotes Requested/Succeed/Failed/Rows: $reqCount/$reqSucceed/$reqFailed/$reqRowsT. Duration: $duration`r`n" | Out-File $logFile -Encoding OEM -Append
$logSummary + ". Quotes Requested/Succeed/Failed/Rows: $reqCount/$reqSucceed/$reqFailed/$reqRowsT. Duration: $duration"

