# 2024-Apr-16. Created by Maxim T. 

$scriptPath = Split-Path -parent $MyInvocation.MyCommand.Definition; 
. ($scriptPath + "\psFunctions.ps1"); # Adding script with reusable functions
. ($scriptPath + "\psSetVariables.ps1"); # Adding script to assign values to common variables

$logFile = $scriptPath + "\Log\" + $MyInvocation.MyCommand.Name.Replace(".ps1", ".txt"); 
(Get-Date).ToString("HH:mm:ss") + " --- Starting script " + $MyInvocation.MyCommand.Name | Out-File $logFile -Encoding OEM; # starting logging to file.
$logSummary = (Get-Date).ToString("HH:mm:ss") + " Script: ExchRate YahooID".PadRight(28);
Get-ChildItem -Path $currExchIDFolder -Filter *Intraday.txt -File | foreach { $_.Delete() }; # Delete existing all files from intraday folder
$currExchIDFile = $currExchIDFolder + "YahooCurrExchIntraday.txt"; #if (Test-Path $currExchIDFile) { Remove-Item $currExchIDFile;} # Removing intraday file before each load
$reqFailed = 0; $reqSucceed = 0; $reqUsed = 0;

$listStart = $config.IndexOf("<Currency>"); $listEnd = $config.IndexOf("</Currency>"); 
if ($listStart -eq -1 -or $listEnd -eq -1 -or $listStart + 1 -ge $listEnd) { "<currency> Symbol list is empty. Exiting script." | Out-File $logFile -Encoding OEM -Append; exit(1); }
$currList = @($config | Select-Object -Index(($listStart + 1)..($listEnd - 1))); #list of symbols we will work on
$currCount = $currList.Count; "Currency count: $currCount. MinDate: $minDate" | Out-File $logFile -Encoding OEM -Append;
if ($currCount -le 1) { "Found just one currency, will not create currency exchange database"; exit(1); }

$urlBase = "https://query2.finance.yahoo.com/v8/finance/chart/@@CURRFROM@@@@CURRTO@@=X?period1=@FromDay@&period2=@ToDay@&interval=1d&events=history"
$toDay = [string] [math]::Floor((get-date -hour 23 -Minute 59 -Second 59 -UFormat %s)); #today in unix timestamp (end of the day)
$fromDay = [string] [math]::Floor((get-Date -hour 0 -Minute 0 -Second 0 -UFormat %s)); #today in unix timestamp (start of the day)
$urlBase = $urlBase.Replace("@FromDay@", $fromDay).Replace("@ToDay@", $toDay);

$logSymbolFolder = $scriptPath + "\Log\SymbolWebpage"; if (!(Test-Path $logSymbolFolder)) { New-Item $logSymbolFolder -type directory }; # setting up folder where text output per currency pair will be temporary stored;

# Use "." for decimal and no 1000s grouping
$culture = [System.Globalization.CultureInfo]::CreateSpecificCulture("en-US")
$culture.NumberFormat.NumberDecimalSeparator = "."
$culture.NumberFormat.NumberGroupSeparator = ""
[System.Threading.Thread]::CurrentThread.CurrentCulture = $culture

$reqUsed = 0; 

For ($currFromIndex = 0; $currFromIndex -lt $currCount; $currFromIndex++) {
    For ($currToIndex = 0; $currToIndex -lt $currCount; $currToIndex++) {
        if ($currList[$currFromIndex] -ne $currList[$currToIndex]) {
            $currFrom = $currList[$currFromIndex]; 
            $currTo = $currList[$currToIndex]; 
            $url = $urlBase.Replace("@@CURRFROM@@", $currFrom).Replace("@@CURRTO@@", $currTo); 
            $currencyPair = $currFrom + "-" + $currTo;
            #$url = $url.Replace("/USD","/"); # if CurrFrom=USD, then remove that (new yahoo requirement 20230118)

            "Requesting url: " + $url | Out-File $logFile -Encoding OEM -Append; 
            try { 
                $reqCount++;
                $wr = Invoke-WebRequest -Uri $url -WebSession $websession -UseBasicParsing; #2025-12-10 Added -UseBasicParsing
            }
            catch { 
                # if attempt to get webpage failed go to next symbol
                $reqFailed++; 
                "  " + $symbol + " - Not Found (web err) `r`n" | Out-File $logFile -Encoding OEM -Append; 
                continue; 
            } 
            if ($wr.StatusCode -ne 200) { 
                $reqFailed++; 
                "  " + $symbol + " - Returned status code: " + $wr.StatusCode + " `r`n" | Out-File $logFile -Encoding OEM -Append; 
                continue; 
            }
            $reqSucceed++; # If reached this point, that means symbol request was successful, go next to copy received data into local file

            $jsonObject = $wr.Content | ConvertFrom-Json

            # Extract required values
            $date = [datetime]::FromFileTimeUtc($jsonObject.chart.result[0].meta.regularMarketTime * 10000000 + 116444736000000000).ToString("yyyy-MM-dd")
            $exchangeRate = $jsonObject.chart.result[0].meta.regularMarketPrice
            $shortName = $jsonObject.chart.result[0].meta.shortName

            # Parse shortName to get currencyFrom and currencyTo
            $currencyPair = $shortName.Split('/')
            $currencyFrom = $currencyPair[0]
            $currencyTo = $currencyPair[1]

            # Create a custom object with the extracted values
            $data = [PSCustomObject]@{
                Date         = $date
                ExchangeRate = $exchangeRate
                CurrencyFrom = $currencyFrom
                CurrencyTo   = $currencyTo
            }

            $ret = GetCurrExchInfo $currFrom $currTo $currExchFolder $minDate; 
            $currExchFile = $ret[0]; 
            $nextDate = $ret[1]; 
            $lastQuote = $ret[2];

            if ($data.Date -ge $nextDate -and $date -gt $lastQuote -and $date -le $todayYMD ) {
                # Export the custom object to a CSV file
                #$data | ConvertTo-Csv -UseQuotes AsNeeded | Select-Object -Skip 1 | Out-File $currExchIDFile -Append -Encoding oem
 
                # Convert the data to CSV format without headers and quotes
                $csvData = $data | ConvertTo-Csv -NoTypeInformation | Select-Object -Skip 1
                $csvData = $csvData -replace '"', ''
	
                # Append the CSV data to the file
                $csvData | Out-File $currExchIDFile -Append -Encoding OEM

                $reqUsed++; 
                $lastQuote = $date
            }
  
            if ($lastQuote -lt $nextDate) { 
                "Done: $currencyPair. No new quotes found. `r`n" | Out-File $logFile -Encoding OEM -Append; 
            } else { 
                "Done: $currencyPair (from: $nextDate). Last quote: $lastQuote rate: $exchangeRate `r`n" | Out-File $logFile -Encoding OEM -Append; 
            }
        }
    }
}

if ( Test-Path $currExchIDFile ) {
    $fc = Get-Item -Path $currExchIDFile; 

    # Removing file when size is 0;
    if ($fc.Length -eq 0) { 
        Remove-Item -Path $currExchIDFile -Force; 
    } 
}

$duration = (NEW-TIMESPAN -Start $startTime -End (Get-Date)).TotalSeconds.ToString("#,##0") + " sec.";
(Get-Date).ToString("HH:mm:ss") + " --- Finished. CurrExch Requested/Succeed/Failed/Used: $reqCount/$reqSucceed/$reqFailed/$reqUsed. Duration: $duration. Date-Time: " + $lastDateTime + "`r`n" | Out-File $logFile -Encoding OEM -append;
$logSummary + ". CurrExch Requested/Succeed/Failed/Used: $reqCount/$reqSucceed/$reqFailed/$reqUsed. Date: " + $lastDateTime + ". Duration: $duration";
