# 2024-Apr-16. Created by Maxim T.,

$scriptPath = Split-Path -parent $MyInvocation.MyCommand.Definition; 
. ($scriptPath + "\psFunctions.ps1");     # Adding script with reusable functions
. ($scriptPath + "\psSetVariables.ps1");  # Adding script to assign values to common variables

$logFile        = $scriptPath + "\Log\" + $MyInvocation.MyCommand.Name.Replace(".ps1",".txt"); 
(Get-Date).ToString("HH:mm:ss") + " --- Starting script " + $MyInvocation.MyCommand.Name | Out-File $logFile -Encoding OEM; # starting logging to file.
$logSummary = (Get-Date).ToString("HH:mm:ss") + " Script: Yahoo Intraday".PadRight(28);
$quoteIDFile = $dataRootFolder + "\QuotesIntraDay\YahooIntraday.txt"; if (Test-Path $quoteIDFile) { Remove-Item $quoteIDFile;} # Removing intraday file before each load

$symbolList = @();
$listStart = $config.IndexOf("<Yahoo>"); $listEnd = $config.IndexOf("</Yahoo>"); 
if ($listStart -eq -1 -or $listEnd -eq -1 -or $listStart+1 -ge $listEnd) {"<Yahoo> Symbol list is empty. " | Out-File $logFile -Encoding OEM -Append; exit(1);}
$symbolList += @($config | Select-Object -Index(($listStart+1)..($listEnd-1))); #list of symbols we will work on

$symbolList = @($symbolList | ?{$a=$_.split(","); ($a[3] -eq $null -or $a[3] -eq "" -or $a[3] -eq "Y") -and ($a[2] -eq "" -or $a[2] -eq $null)}); # by default if value is empty or y, then get intra-day quotes. Also MaxDate for symbol should not be specified
"Symbol count: " + $symbolList.count + ". MinDate: $minDate" | Out-File $logFile -Encoding OEM -Append;
$urlBase = "https://query1.finance.yahoo.com/v7/finance/download/@Symbol@?period1=@FromDay@&period2=@ToDay@&interval=1d&events=history&includeAdjustedClose=true&crumb=@CRUMB@"
$toDay   = [string] [math]::Floor((get-date -hour 23 -Minute 59 -Second 59 -UFormat %s)); #today in unix timestamp (end of the day)
$fromDay = [string] [math]::Floor((get-Date -hour 0 -Minute 0 -Second 0 -UFormat %s));    #today in unix timestamp (start of the day)
$urlBase = $urlBase.Replace("@CRUMB@", $crumb).Replace("@ToDay@", $toDay).Replace("@FromDay@", $fromDay); # From and To dates are the same - todays date

$logSymbolFolder = $scriptPath + "\Log\SymbolWebpage"; if (!(Test-Path $logSymbolFolder)) {New-Item $logSymbolFolder -type directory}; # setting up folder where text output per symbol will be temporary stored;

ForEach($sLine in $symbolList) { # For each symbol
    $ret = GetSymbolInfo $sLine $quotesFolder $minDate; $symbol = $ret[0]; $nextDate = $ret[1]; $symbolMaxDate = $ret[2]; $symbolMaxDateAdj = $ret[3]; $symbolQuoteFile = $ret[4]; $lastQuote = $ret[5];
    "Symbol: " + $symbol.PadLeft(12) + ". Next date: $nextDate. Quote file: $symbolQuoteFile" | Out-File $logFile -Encoding OEM -Append; 
    if ($nextDate -gt $symbolMaxDate) { "  We already have data up to maximum configuration date of $symbolMaxDate. Will not request new data for this symbol." | Out-File $logFile -Encoding OEM -Append; continue; } 

     # ========== Get latest quote from website - Start ==============
     $url = $urlBase.Replace("@Symbol@",$symbol);
    "  Requesting url: " + $url | Out-File $logFile -Encoding OEM -Append; $webpage = ""; $reqRows=0; 
    try {$reqCount++; $wr = Invoke-WebRequest -Uri $url -WebSession $websession; }
    catch { $reqFailed++; "  " + $symbol + " - Not Found (web err) `r`n" | Out-File $logFile -Encoding OEM -Append; continue; } # if attempt to get webpage failed go to next symbol
    if ($wr.StatusCode -ne 200) {$reqFailed++; "  " + $symbol + " - Returned status code: " + $wr.StatusCode + " `r`n" | Out-File $logFile -Encoding OEM -Append; continue;  }

    $quotesTxt = $wr.Content; # This variable now contains downloaded quotes in text format:    Date,Open,High,Low,Close,Adj Close,Volume    |   2017-07-17,78.830002,78.930000,78.760002,78.839996,78.839996,472700
    if ($quotesTxt.length -le 45) { $reqFailed++; "  " + $symbol + " - Not Found (return empty file) `r`n" | Out-File $logFile -Encoding OEM -Append; continue;} # Check if data received makes sense. If request size is less than 45 bytes (header length is 42), then there is something wrong with data received. Ignoring it, going to next symbol
    if ($quotesTxt.Contains("<html>")) {$reqFailed++; "  " + $symbol + " - Not Found (return html) `r`n" | Out-File $logFile -Encoding OEM -Append; continue;} # Result is html file, ignore such file
    $reqSucceed++; # If reached this point, that means symbol request was successful, go next to copy received data into local file
    # ========== Get quotes from website - End. Quote is in $quotesTxt ==============

    $ql = $quotesTxt.Split("`n") | ? {$_.trim() -ne ""  -and ($_.StartsWith("1") -or $_.StartsWith("2"))} | Sort-Object -Descending; # Sometimes duplicate records could come back, need to sort records and not load duplicate values
    if ($ql.Count -ge 2) {$ql=$ql[0];}; #if more than one record come back, just use first record

    $ql | %{$a=$_.Split(","); if($a[0] -ge $nextDate -and $a[0] -gt $lastQuote -and $a[0] -le $symbolMaxDateAdj -and $a[0] -le $todayYMD -and $a[4] -match "^[\d\.]+$") {$a[0]+","+$a[4]+","+$symbol; $reqRows++; $lastQuote=$a[0]; } }  |
       Out-File $quoteIDFile -Encoding OEM -Append #taking just date,Close,Symbol
    if ($lastQuote -lt $nextDate) {"Done: $symbol. No new quotes found. `r`n" | Out-File $logFile -Encoding OEM -Append;} # No new records where loaded
    else {"Done: $symbol (from: $nextDate). Record count: $reqRows. Last quote: $lastQuote `r`n" | Out-File $logFile -Encoding OEM -Append; $reqRowsT+=$reqRows;}
}
$fc = Get-Item -Path $quoteIDFile; if ($fc.Length -eq 0) {Remove-Item -Path $quoteIDFile;} # Removing file when size is 0;

$duration = (NEW-TIMESPAN -Start $startTime -End (Get-Date)).TotalSeconds.ToString("#,##0") + " sec.";
(Get-Date).ToString("HH:mm:ss") + " --- Finished. Quotes Requested/Succeed/Failed/Rows: $reqCount/$reqSucceed/$reqFailed/$reqRowsT. Duration: $duration`r`n" | Out-File $logFile -Encoding OEM -append;
$logSummary + ". Quotes Requested/Succeed/Failed/Rows: $reqCount/$reqSucceed/$reqFailed/$reqRowsT. Duration: $duration";
